<?php
/**
 * PrestaShop module created by VEKIA, a guy from official PrestaShop community ;-)
 *
 * @author    VEKIA PL MIŁOSZ MYSZCZUK VATEU: PL9730945634
 * @copyright 2010-2026 VEKIA
 * @license   This program is not free software and you can't resell and redistribute it
 *
 * CONTACT WITH DEVELOPER http://mypresta.eu
 * support@mypresta.eu
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Class cmshomepage
 * Main module class for displaying CMS content on the homepage.
 */
class cmshomepage extends Module
{
    /** @var string License key for updates */
    public $mkey;

    /** @var string Link to the module's official website */
    public $mypresta_link;

    /**
     * cmshomepage constructor.
     * Initializes module metadata and the update system.
     */
    public function __construct()
    {
        $this->name = 'cmshomepage';
        $this->tab = 'front_office_features';
        $this->author = 'MyPresta.eu';
        $this->version = '2.0.0';
        $this->bootstrap = true;
        $this->mypresta_link = 'https://mypresta.eu/modules/front-office-features/cms-on-homepage.html';

        parent::__construct();

        $this->displayName = $this->l('CMS on homepage');
        $this->description = $this->l('Insert CMS content to your homepage');

        // Delegate update check system initialization to the helper class
        cmshomepageUpdate::setUp($this);
    }

    /**
     * Installs the module and registers necessary hooks.
     *
     * @return bool
     */
    public function install()
    {
        if (parent::install() == false ||
            $this->registerHook('displayHome') == false ||
            $this->registerHook('displayHomeTab') == false ||
            $this->registerHook('displayHomeTabContent') == false ||
            !Configuration::updateValue('update_' . $this->name, '0') ||
            !Configuration::updateValue('mypresta_updates', 1) ||
            !Configuration::updateValue('cmshomepage', '0')) {
            return false;
        }
        return true;
    }

    /**
     * Checks for structural inconsistencies.
     *
     * @return string
     */
    public function inconsistency()
    {
        return cmshomepageUpdate::inconsistency();
    }

    /**
     * Checks for new versions of the module.
     *
     * @param int $display_msg If 1, returns a formatted status string.
     * @param int $form If 1, returns the update settings form.
     * @return string
     */
    public function checkforupdates($display_msg = 0, $form = 0)
    {
        return cmshomepageUpdate::checkforupdates($this, $display_msg, $form);
    }

    /**
     * Handles the module configuration page in Back Office.
     *
     * @return string
     */
    public function getContent()
    {
        $output = '';

        // Process settings submission
        if (Tools::isSubmit('submitcmshomepage')) {
            $id_cms = (int)Tools::getValue('cmshomepage');
            Configuration::updateValue('cmshomepage', $id_cms);
            $output .= $this->displayConfirmation($this->l('Settings updated successfully.'));
        }

        return $output . $this->advert() . $this->displayForm() . $this->checkforupdates(0, 1);
    }

    /**
     * Renders advertising and external information frames.
     *
     * @return string
     */
    public function advert()
    {
        return '<iframe src="//dev.addons.ovh/modules.html?module='.$this->name.'" width="100%" height="150" border="0" style="border:none;"></iframe>';
    }

    /**
     * Generates the module configuration form using HelperForm.
     *
     * @return string
     */
    public function displayForm()
    {
        $id_lang = (int)Configuration::get('PS_LANG_DEFAULT');
        $cms_pages = CMS::listCms($id_lang);

        $options = array(
            array('id_cms' => 0, 'meta_title' => $this->l('-- SELECT --'))
        );

        foreach ($cms_pages as $page) {
            $options[] = array(
                'id_cms' => $page['id_cms'],
                'meta_title' => $page['meta_title']
            );
        }

        $fields_form = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('CMS Content Settings'),
                    'icon' => 'icon-cogs'
                ),
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Select CMS Page'),
                        'name' => 'cmshomepage',
                        'desc' => $this->l('Choose the CMS page whose content you want to display on your homepage. Make sure the page is active and has content.'),
                        'options' => array(
                            'query' => $options,
                            'id' => 'id_cms',
                            'name' => 'meta_title'
                        )
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save Configuration'),
                    'class' => 'btn btn-default pull-right'
                )
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->name;
        $lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitcmshomepage';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => array(
                'cmshomepage' => Tools::getValue('cmshomepage', Configuration::get('cmshomepage')),
            ),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id
        );

        return $helper->generateForm(array($fields_form));
    }

    /**
     * Checks the PrestaShop version and returns specific parts or comparison results.
     *
     * @param mixed $part_or_version Part index or version string to compare.
     * @param string $operator Operator for version_compare.
     * @return bool|int
     */
    public function psversion($part_or_version = 1, $operator = '>')
    {
        if (is_string($part_or_version) && strpos($part_or_version, '.') !== false) {
            return version_compare(_PS_VERSION_, $part_or_version, $operator);
        }
        $exp = explode('.', _PS_VERSION_);
        $offset = ($exp[0] === '1') ? 1 : 0;
        $index = $offset + $part_or_version - 1;
        return isset($exp[$index]) ? (int)$exp[$index] : 0;
    }

    /**
     * Hook to display content in the home page main area.
     *
     * @param array $params
     * @return string
     */
    public function hookDisplayHome($params)
    {
        $id_cms = (int)Configuration::get('cmshomepage');
        if (!$id_cms) {
            return '';
        }

        $cms = new CMS($id_cms, $this->context->language->id);
        if (!Validate::isLoadedObject($cms)) {
            return '';
        }

        $this->context->smarty->assign('cmsonhome', $cms);
        return $this->display(__FILE__, 'cmshomepage.tpl');
    }

    /**
     * Hook to display the tab link on the homepage (PS 1.6).
     *
     * @param array $params
     * @return string
     */
    public function hookDisplayHomeTab($params)
    {
        $id_cms = (int)Configuration::get('cmshomepage');
        if (!$id_cms) {
            return '';
        }

        $cms = new CMS($id_cms, $this->context->language->id);
        if (!Validate::isLoadedObject($cms)) {
            return '';
        }

        $this->context->smarty->assign('cmsonhome', $cms);
        return $this->display(__FILE__, 'tab.tpl');
    }

    /**
     * Hook to display the tab content on the homepage (PS 1.6).
     *
     * @param array $params
     * @return string
     */
    public function hookDisplayHomeTabContent($params)
    {
        $id_cms = (int)Configuration::get('cmshomepage');
        if (!$id_cms) {
            return '';
        }

        $cms = new CMS($id_cms, $this->context->language->id);
        if (!Validate::isLoadedObject($cms)) {
            return '';
        }

        $this->context->smarty->assign('cmsonhome', $cms);
        return $this->display(__FILE__, 'tabContents.tpl');
    }
}

/**
 * Class cmshomepageUpdate
 * Helper class for handling module updates via MyPresta.eu API.
 */
class cmshomepageUpdate
{
    /**
     * Sets up the update system, checks for license keys and verifies version.
     *
     * @param cmshomepage $module
     * @return void
     */
    public static function setUp($module)
    {
        $module->mkey = "nlc";
        $key_path = dirname(__FILE__) . '/key.php';
        if (@file_exists($key_path)) {
            @require_once($key_path);
        }

        if (defined('_PS_ADMIN_DIR_')) {
            $update_interval = 259200; // 3 days
            $last_update = (int)Configuration::get('update_' . $module->name);

            if (Configuration::get('mypresta_updates') != 0) {
                if ($last_update < (time() - $update_interval)) {
                    self::verify($module->name, (isset($module->mkey) ? $module->mkey : 'nokey'), $module->version);
                }

                $available_version = Configuration::get('updatev_' . $module->name);
                if (version_compare($module->version, $available_version, '<') && Tools::getValue('ajax', 'false') == 'false') {
                    $context = Context::getContext();
                    if (isset($context->controller)) {
                        $context->controller->warnings[] = '<strong>' . $module->displayName . '</strong>: ' . $module->l('New version available, check http://MyPresta.eu for more informations') . ' <a href="' . $module->mypresta_link . '">' . $module->l('More details in changelog') . '</a>';
                        $module->warning = $context->controller->warnings[0];
                    }
                }
            } else {
                if ($last_update < (time() - $update_interval)) {
                    self::verify($module->name, (isset($module->mkey) ? $module->mkey : 'nokey'), $module->version);
                }
            }
        }
    }

    /**
     * Placeholder for consistency checks.
     *
     * @return string
     */
    public static function inconsistency()
    {
        return '';
    }

    /**
     * Renders update notification or the update settings form.
     *
     * @param cmshomepage $module
     * @param int $display_msg
     * @param int $form
     * @return string
     */
    public static function checkforupdates($module, $display_msg = 0, $form = 0)
    {
        $output = '';

        if (Tools::isSubmit('submit_settings_updates')) {
            Configuration::updateValue('mypresta_updates', (int)Tools::getValue('mypresta_updates'));
            $output .= $module->displayConfirmation($module->l('Update settings saved.'));
        }

        if ($form == 1) {
            return $output . self::renderUpdateForm($module);
        } else {
            if ($display_msg == 1) {
                $available_version = Configuration::get('updatev_' . $module->name);
                if (version_compare($module->version, $available_version, '<')) {
                    return "<span style='color:red; font-weight:bold; font-size:16px; margin-right:10px;'>" . $module->l('New version available!') . "</span>";
                } else {
                    return "<span style='color:green; font-weight:bold; font-size:16px; margin-right:10px;'>" . $module->l('Module is up to date!') . "</span>";
                }
            }
        }
        return $output;
    }

    /**
     * Builds the update settings form using HelperForm.
     *
     * @param cmshomepage $module
     * @return string
     */
    private static function renderUpdateForm($module)
    {
        $context = Context::getContext();
        $msg = '';
        if (Tools::isSubmit('submit_settings_updates_now')) {
            self::verify($module->name, (isset($module->mkey) ? $module->mkey : 'nokey'), $module->version);
            $msg = self::checkforupdates($module, 1, 0);
        }

        $fields_form = array(
            'form' => array(
                'legend' => array(
                    'title' => $module->l('MyPresta Updates'),
                    'icon' => 'icon-wrench'
                ),
                'input' => array(
                    array(
                        'type' => 'free',
                        'label' => $module->l('Manual Check'),
                        'name' => 'check_updates_btn',
                        'desc' => $module->l('Manually check if a new version of this module is available on MyPresta.eu servers.')
                    ),
                    array(
                        'type' => 'select',
                        'label' => $module->l('Automatic Notifications'),
                        'name' => 'mypresta_updates',
                        'desc' => $module->l('Enable or disable automatic check for module updates. If enabled, the module will notify you in the Back Office when a new version is released.'),
                        'options' => array(
                            'query' => array(
                                array('id' => '1', 'name' => $module->l('Enabled')),
                                array('id' => '0', 'name' => $module->l('Disabled')),
                            ),
                            'id' => 'id',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'free',
                        'label' => $module->l('Product Page'),
                        'name' => 'module_page_link',
                        'desc' => $module->l('Direct link to the official module page on MyPresta.eu.')
                    ),
                ),
                'submit' => array(
                    'title' => $module->l('Save Update Settings'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submit_settings_updates'
                )
            )
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $module->name;
        $lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->identifier = 'id_module';
        $helper->submit_action = 'submit_settings_updates';
        $helper->currentIndex = $context->link->getAdminLink('AdminModules', false) . '&configure=' . $module->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $check_btn = '<button type="submit" name="submit_settings_updates_now" class="btn btn-default"><i class="process-icon-update"></i> ' . $module->l('Check Now') . '</button>';

        $helper->tpl_vars = array(
            'fields_value' => array(
                'check_updates_btn' => $msg . $check_btn,
                'mypresta_updates' => Configuration::get('mypresta_updates'),
                'module_page_link' => '<a href="' . $module->mypresta_link . '" target="_blank">' . $module->displayName . '</a>'
            ),
            'languages' => $context->controller->getLanguages(),
            'id_language' => $context->language->id
        );

        return $helper->generateForm(array($fields_form));
    }

    /**
     * Communicates with MyPresta.eu API to verify the module version.
     *
     * @param string $module_name
     * @param string $key
     * @param string $version
     * @return string
     */
    public static function verify($module_name, $key, $version)
    {
        $actual_version = '';
        if (ini_get("allow_url_fopen")) {
            $url = 'http://dev.mypresta.eu/update/get.php?module=' . $module_name
                . "&version=" . base64_encode($version)
                . "&lic=" . $key
                . "&u=" . base64_encode(_PS_BASE_URL_ . __PS_BASE_URI__);
            $actual_version = @file_get_contents($url);
        }

        Configuration::updateValue("update_" . $module_name, (string)time());
        Configuration::updateValue("updatev_" . $module_name, (string)$actual_version);
        return (string)$actual_version;
    }
}